/*------------------------------------------------------------------
			Maths Library
			-------------

	Header for C implementation of basic geometry related
	maths functions

	(C) Silicon Dream Ltd 1994

  ------------------------------------------------------------------

Changes:						Date:
* Created file						17/08/94
*/

#ifndef MATHS
#define MATHS

#include <atomic.h>

/* Comonly used constants */

#define PI		3.141592654
#define TWO_PI		(2.0*PI)
#define HALF_PI		(0.5*PI)
#define DEG2RAD(x)	((float) (x)*TWO_PI/360.0)
#define RAD2DEG(x)	((float) (x)*360.0/TWO_PI)
#define ZERO_TOL	0.000001

/* General typedefs */

typedef struct Vectortag		/* vec */
	{
	float		fX;
	float		fY;
	float		fZ;
	} Vector;

typedef struct LongVectag		/* lvec */
	{
	float		fX;
	float		fY;
	float		fZ;
	float		fW;
	} LongVec;

/* For performance reasons (see _dyn MthTransLVec routine), we ensure that when we
   store data in a matrix, we must treat the first index as a column index */

typedef float		Matrix[4][4];	/* mat */
typedef const float	CMatrix[4][4];	/* mat */

/* Functions */

float _dyn MthSqr (float fN);
void _dyn MthConvToPolar (const Vector *pvec, float *pfTheta,
			  float *pfPhi, float *pfRho);
void _dyn MthConvToCart (Vector *pvec, float fTheta,
			 float fPhi, float fRho);
void _dyn MthMakeVec (Vector *pvec, float fX, float fY, float fZ);
bool _dyn MthVecsEqual (const Vector *pvecA, const Vector *pvecB);
void _dyn MthMakeLVec (LongVec *plvec, float fX, float fY,
		       float fZ, float fW);
bool _dyn MthLVecsEqual (const LongVec *plvecA, const LongVec *plvecB);
void _dyn MthTransVec (Vector *pvec, CMatrix mat);
void _dyn MthTransLVec (LongVec *plvec, CMatrix mat);
void _dyn MthTransMatrix (CMatrix matA, CMatrix matB, Matrix matRes);
void _dyn MthAddVec (const Vector *pvecA, const Vector *pvecB, Vector *pvecRes);
void _dyn MthSubVec (const Vector *pvecA, const Vector *pvecB, Vector *pvecRes);
void _dyn MthScaleVec (Vector *pvec, float fScale);
float _dyn MthLenVec (const Vector *pvec);
void _dyn MthCrossProd (const Vector *pvecA, const Vector *pvecB, Vector *pvecRes);
float _dyn MthDotProd (const Vector *pvecA, const Vector *pvecB);
bool _dyn MthMatsEqual (CMatrix matA, CMatrix matB);
void _dyn MthSetMat (Matrix mat, float f1, float f2, float f3, float f4, float f5, float f6, float f7, float f8, float f9, float f10, float f11, float f12, float f13, float f14, float f15, float f16);
void _dyn MthGetIdentMat (Matrix mat);
void _dyn MthGetXrotMat (Matrix mat, float fAng);
void _dyn MthGetYrotMat (Matrix mat, float fAng);
void _dyn MthGetZrotMat (Matrix mat, float fAng);
void _dyn MthGetTranslMat (Matrix mat, const Vector *pvec);
void _dyn MthGetScaleMat (Matrix mat, float fX, float fY, float fZ);
bool _dyn MthCompInvMat (CMatrix mat, Matrix matInv);
void _dyn MthCompTransMat (const Vector *pvecOrg, const Vector *pvecAxZ,
			   const Vector *pvecAxY, Matrix matT);
void _dyn MthTranslParent (const Vector *pvec, Matrix matTrans);
void _dyn MthTranslChild (const Vector *pvec, Matrix matTrans);
void _dyn MthScaleParent (float fX, float fY, float fZ, Matrix matTrans);
void _dyn MthScaleChild (float fX, float fY, float fZ, Matrix matTrans);
void _dyn MthXrotParent (float fAng, Matrix matTrans);
void _dyn MthYrotParent (float fAng, Matrix matTrans);
void _dyn MthZrotParent (float fAng, Matrix matTrans);
void _dyn MthXrotChild (float fAng, Matrix matTrans);
void _dyn MthYrotChild (float fAng, Matrix matTrans);
void _dyn MthZrotChild (float fAng, Matrix matTrans);
void _dyn MthReverseX (Matrix matTrans);
void _dyn MthReverseY (Matrix matTrans);
void _dyn MthReverseZ (Matrix matTrans);

#endif			// Do not include this file twice
